/*
 * Copyright (c) 2001, 2018, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package nsk.jdi.MethodExitRequest.addClassExclusionFilter;

import nsk.share.*;
import nsk.share.jpda.*;
import nsk.share.jdi.*;

import com.sun.jdi.*;
import com.sun.jdi.event.*;
import com.sun.jdi.request.*;

import java.util.*;
import java.io.*;

/**
 * The test for the implementation of an object of the type
 * MethodExitRequest.
 *
 * The test checks that results of the method
 * <code>com.sun.jdi.MethodExitRequest.addClassExclusionFilter()</code>
 * complies with its spec.
 *
 * The test checks up on the following assertion:
 *    Restricts the events generated by this request to those
 *    whose method is in a class whose name does not match
 *    this restricted regular expression. e.g. "java.*" or "*.Foo".
 * The case to checks a pattern that ends with '*'.
 *
 * The test works as follows.
 * - The debugger
 *   - sets up two MethodExitRequests,
 *   - restricts the Requests using patterns 'java*' and 'sun*',
 *     so that events will be filtered only from test classes,
 *   - resumes the debuggee, and
 *   - waits for expected MethodExitEvents.
 * - The debuggee creates and starts thread1, which being run,
 *   invoke methods used
 *   to generate Events and to test the filters.
 * - Upon getting the events, the debugger performs checks required.
 */

public class filter001 extends TestDebuggerType1 {

    public static void main (String argv[]) {
        System.exit(run(argv, System.out) + Consts.JCK_STATUS_BASE);
    }

    public static int run (String argv[], PrintStream out) {
        debuggeeName = "nsk.jdi.MethodExitRequest.addClassExclusionFilter.filter001a";
        return new filter001().runThis(argv, out);
    }

    private String classExclName1 = "java";
    private String classExclName2 = "sun";
    private boolean methodExitReceived = false;

    protected void testRun() {

        MethodExitRequest eventRequest1 = null;
        String thread1Name = "thread1";
        String property1 = "MethodExitRequest1";

        for (int i = 0; ; i++) {

            if (!shouldRunAfterBreakpoint()) {
                vm.resume();
                break;
            }

            display(":::::: case: # " + i);

            switch (i) {

                case 0:
                ThreadReference thread1 = (ThreadReference) debuggeeClass.getValue(
                                                  debuggeeClass.fieldByName(thread1Name));

                eventRequest1 = setting23MethodExitRequest( thread1,
                                                           EventRequest.SUSPEND_NONE,
                                                           property1);

                eventRequest1.addClassExclusionFilter(classExclName1 + "*");
                eventRequest1.addClassExclusionFilter(classExclName2 + "*");
                eventRequest1.enable();

                eventHandler.addListener(
                     new EventHandler.EventListener() {
                         public boolean eventReceived(Event event) {
                            if (event instanceof MethodExitEvent) {
                                methodExitReceived = true;
                                String str = ((MethodExitEvent)event).location().declaringType().name();
                                if (str.indexOf(classExclName1) == 0 || str.indexOf(classExclName2) == 0) {
                                    setFailedStatus("Received unexpected MethodExitEvent for excluded class:" + str);
                                } else {
                                    display("Received expected MethodExitEvent for " + str);
                                }
                                return true;
                            }
                            return false;
                         }
                     }
                );

                display("......waiting for MethodExitEvent in expected thread");
                vm.resume();
                break;

                default:
                throw new Failure("** default case 1 **");
            }
        }
        if (!methodExitReceived) {
            setFailedStatus("No MethodExitEvent was received.");
        }
        return;
    }

    private MethodExitRequest setting23MethodExitRequest ( ThreadReference thread,
                                                           int             suspendPolicy,
                                                           String          property       ) {
        try {
            display("......setting up MethodExitRequest:");
            display("       thread: " + thread + ";  property: " + property);

            MethodExitRequest
            menr = eventRManager.createMethodExitRequest();
            menr.putProperty("number", property);
            if (thread != null)
                menr.addThreadFilter(thread);
            menr.setSuspendPolicy(suspendPolicy);

            display("      a MethodExitRequest has been set up");
            return menr;
        } catch ( Exception e ) {
            throw new Failure("** FAILURE to set up MethodExitRequest **");
        }
    }
}
