/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.webadmin.service;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;
import jakarta.inject.Inject;
import java.time.Clock;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.james.core.Username;
import org.apache.james.mailbox.MailboxManager;
import org.apache.james.mailbox.MailboxSession;
import org.apache.james.mailbox.MessageManager;
import org.apache.james.mailbox.MessageUid;
import org.apache.james.mailbox.exception.MailboxException;
import org.apache.james.mailbox.exception.MailboxNotFoundException;
import org.apache.james.mailbox.model.FetchGroup;
import org.apache.james.mailbox.model.MailboxPath;
import org.apache.james.mailbox.model.MessageRange;
import org.apache.james.mailbox.model.MessageResult;
import org.apache.james.mailbox.model.SearchQuery;
import org.apache.james.task.Task;
import org.apache.james.user.api.UsersRepository;
import org.apache.james.user.api.UsersRepositoryException;
import org.apache.james.util.DurationParser;
import org.apache.james.util.ReactorUtils;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public class ExpireMailboxService {
    private static final Logger LOGGER = LoggerFactory.getLogger(ExpireMailboxService.class);
    private final UsersRepository usersRepository;
    private final MailboxManager mailboxManager;
    private final MessageListingStrategy.Factory factory;

    @Inject
    public ExpireMailboxService(UsersRepository usersRepository, MailboxManager mailboxManager, Clock clock) {
        this.usersRepository = usersRepository;
        this.mailboxManager = mailboxManager;
        this.factory = new MessageListingStrategy.Factory(clock);
    }

    public Mono<Task.Result> expireMailboxes(Context context, RunningOptions runningOptions) {
        return this.users(runningOptions).transform(ReactorUtils.throttle().elements(runningOptions.getUsersPerSecond()).per(Duration.ofSeconds(1L)).forOperation(username -> this.loadMailbox(runningOptions, (Username)username).flatMap(mailboxWithSession -> this.expireMessages(context, runningOptions, (Pair<MessageManager, MailboxSession>)mailboxWithSession)))).reduce((Object)Task.Result.COMPLETED, Task::combine).onErrorResume(UsersRepositoryException.class, e -> {
            LOGGER.error("Error while accessing users from repository", (Throwable)e);
            return Mono.just((Object)Task.Result.PARTIAL);
        });
    }

    private Flux<Username> users(RunningOptions runningOptions) {
        return runningOptions.user.map(Username::of).map(Flux::just).orElseGet(() -> Flux.from((Publisher)this.usersRepository.listReactive()));
    }

    private Mono<Pair<MessageManager, MailboxSession>> loadMailbox(RunningOptions runningOptions, Username username) {
        MailboxSession session = this.mailboxManager.createSystemSession(username);
        MailboxPath mailboxPath = MailboxPath.forUser((Username)username, (String)runningOptions.getMailbox());
        return Mono.from((Publisher)this.mailboxManager.getMailboxReactive(mailboxPath, session)).onErrorResume(MailboxNotFoundException.class, ignore -> Mono.empty()).map(messageManager -> Pair.of((Object)messageManager, (Object)session));
    }

    private Mono<Task.Result> expireMessages(Context context, RunningOptions runningOptions, Pair<MessageManager, MailboxSession> mailboxWithSession) {
        return this.factory.choose(runningOptions).listMessages((MessageManager)mailboxWithSession.getKey(), runningOptions, (MailboxSession)mailboxWithSession.getValue()).flatMap(uids -> this.deleteMessagesReactive((MessageManager)mailboxWithSession.getKey(), (MailboxSession)mailboxWithSession.getValue(), (List<MessageUid>)uids)).doOnNext(expired -> this.recordSuccess(context, (Integer)expired)).then(Mono.just((Object)Task.Result.COMPLETED)).onErrorResume(e -> this.manageFailure(context, mailboxWithSession, (Throwable)e));
    }

    private Mono<Integer> deleteMessagesReactive(MessageManager mgr, MailboxSession session, List<MessageUid> uids) {
        if (uids.isEmpty()) {
            return Mono.just((Object)0);
        }
        return Flux.fromIterable(uids).window(128).flatMap(Flux::collectList).concatMap(u -> mgr.deleteReactive(u, session).thenReturn((Object)u.size())).reduce(Integer::sum);
    }

    private Mono<Task.Result> manageFailure(Context context, Pair<MessageManager, MailboxSession> mailboxWithSession, Throwable e) {
        LOGGER.warn("Failed to expire user mailbox {}", (Object)((MailboxSession)mailboxWithSession.getValue()).getUser().asString(), (Object)e);
        context.incrementFailedCount();
        context.incrementProcessedCount();
        return Mono.just((Object)Task.Result.PARTIAL);
    }

    private void recordSuccess(Context context, Integer expired) {
        if (expired > 0) {
            context.incrementExpiredCount();
            context.incrementMessagesDeleted(expired.intValue());
        }
        context.incrementProcessedCount();
    }

    private static interface MessageListingStrategy {
        public Mono<List<MessageUid>> listMessages(MessageManager var1, RunningOptions var2, MailboxSession var3);

        public static class Factory {
            private final Clock clock;

            public Factory(Clock clock) {
                this.clock = clock;
            }

            MessageListingStrategy choose(RunningOptions runningOptions) {
                if (runningOptions.byExpiresHeader) {
                    return new HeaderBasedMessageListingStrategy(this.clock);
                }
                Preconditions.checkArgument((boolean)runningOptions.maxAgeDuration.isPresent());
                if (runningOptions.getUseSavedDate().equals(Optional.of(true))) {
                    return new SavedDateDateBeforeListingStrategy(this.clock);
                }
                return new InternalDateBeforeListingStrategy(this.clock);
            }
        }

        public static class HeaderBasedMessageListingStrategy
        implements MessageListingStrategy {
            private final Clock clock;

            public HeaderBasedMessageListingStrategy(Clock clock) {
                this.clock = clock;
            }

            @Override
            public Mono<List<MessageUid>> listMessages(MessageManager messageManager, RunningOptions runningOptions, MailboxSession session) {
                SearchQuery.Criterion criterion = SearchQuery.headerDateBefore((String)"Expires", (Date)Date.from(this.clock.instant()), (SearchQuery.DateResolution)SearchQuery.DateResolution.Second);
                try {
                    return Flux.from((Publisher)messageManager.search(SearchQuery.of((SearchQuery.Criterion[])new SearchQuery.Criterion[]{criterion}), session)).collectList();
                }
                catch (MailboxException e) {
                    return Mono.error((Throwable)e);
                }
            }
        }

        public static class SavedDateDateBeforeListingStrategy
        implements MessageListingStrategy {
            private final Clock clock;

            public SavedDateDateBeforeListingStrategy(Clock clock) {
                this.clock = clock;
            }

            @Override
            public Mono<List<MessageUid>> listMessages(MessageManager messageManager, RunningOptions runningOptions, MailboxSession session) {
                Preconditions.checkArgument((boolean)runningOptions.maxAgeDuration.isPresent());
                Preconditions.checkArgument((boolean)runningOptions.getUseSavedDate().equals(Optional.of(true)));
                Date limit = Date.from(this.clock.instant().minus(runningOptions.maxAgeDuration.get()));
                return Flux.from((Publisher)messageManager.getMessagesReactive(MessageRange.all(), FetchGroup.MINIMAL, session)).filter(message -> message.getSaveDate().map(savedDate -> savedDate.compareTo(limit) < 0).orElse(true)).map(MessageResult::getUid).collectList();
            }
        }

        public static class InternalDateBeforeListingStrategy
        implements MessageListingStrategy {
            private final Clock clock;

            public InternalDateBeforeListingStrategy(Clock clock) {
                this.clock = clock;
            }

            @Override
            public Mono<List<MessageUid>> listMessages(MessageManager messageManager, RunningOptions runningOptions, MailboxSession session) {
                Preconditions.checkArgument((boolean)runningOptions.maxAgeDuration.isPresent());
                Date limit = Date.from(this.clock.instant().minus(runningOptions.maxAgeDuration.get()));
                SearchQuery.Criterion internalDateBefore = SearchQuery.internalDateBefore((Date)limit, (SearchQuery.DateResolution)SearchQuery.DateResolution.Second);
                try {
                    return Flux.from((Publisher)messageManager.search(SearchQuery.of((SearchQuery.Criterion[])new SearchQuery.Criterion[]{internalDateBefore}), session)).collectList();
                }
                catch (MailboxException e) {
                    return Mono.error((Throwable)e);
                }
            }
        }
    }

    public static class RunningOptions {
        public static final RunningOptions DEFAULT = new RunningOptions(1, "INBOX", Optional.of(false), Optional.empty(), true, Optional.empty());
        private final int usersPerSecond;
        private final String mailbox;
        private final boolean byExpiresHeader;
        private final Optional<String> olderThan;
        private final Optional<Boolean> useSavedDate;
        private final Optional<String> user;
        @JsonIgnore
        private final Optional<Duration> maxAgeDuration;

        public static RunningOptions fromParams(Optional<String> byExpiresHeader, Optional<String> olderThan, Optional<String> usersPerSecond, Optional<String> mailbox, boolean useSavedDate, Optional<String> user) {
            try {
                if (byExpiresHeader.isPresent() == olderThan.isPresent()) {
                    throw new IllegalArgumentException("Must specify either 'olderThan' or 'byExpiresHeader' parameter");
                }
                return new RunningOptions(usersPerSecond.map(Integer::parseInt).orElse(DEFAULT.getUsersPerSecond()), mailbox.orElse(DEFAULT.getMailbox()), Optional.of(useSavedDate), user, byExpiresHeader.isPresent(), olderThan);
            }
            catch (NumberFormatException ex) {
                throw new IllegalArgumentException("'usersPerSecond' must be numeric");
            }
        }

        @JsonCreator
        public RunningOptions(@JsonProperty(value="usersPerSecond") int usersPerSecond, @JsonProperty(value="mailbox") String mailbox, @JsonProperty(value="useSavedDate") Optional<Boolean> useSavedDate, @JsonProperty(value="user") Optional<String> user, @JsonProperty(value="byExpiresHeader") boolean byExpiresHeader, @JsonProperty(value="olderThan") Optional<String> olderThan) {
            Preconditions.checkArgument((usersPerSecond > 0 ? 1 : 0) != 0, (Object)"'usersPerSecond' must be strictly positive");
            this.usersPerSecond = usersPerSecond;
            this.mailbox = mailbox;
            this.byExpiresHeader = byExpiresHeader;
            this.useSavedDate = useSavedDate;
            this.user = user;
            this.olderThan = olderThan;
            this.maxAgeDuration = olderThan.map(v -> DurationParser.parse((String)((String)olderThan.get()), (ChronoUnit)ChronoUnit.DAYS));
        }

        public int getUsersPerSecond() {
            return this.usersPerSecond;
        }

        public String getMailbox() {
            return this.mailbox;
        }

        public boolean getByExpiresHeader() {
            return this.byExpiresHeader;
        }

        public Optional<String> getOlderThan() {
            return this.olderThan;
        }

        public Optional<Boolean> getUseSavedDate() {
            return this.useSavedDate;
        }

        public Optional<String> getUser() {
            return this.user;
        }
    }

    public static class Context {
        private final AtomicLong inboxesExpired = new AtomicLong(0L);
        private final AtomicLong inboxesFailed = new AtomicLong(0L);
        private final AtomicLong inboxesProcessed = new AtomicLong(0L);
        private final AtomicLong messagesDeleted = new AtomicLong(0L);

        public long getInboxesExpired() {
            return this.inboxesExpired.get();
        }

        public long getInboxesFailed() {
            return this.inboxesFailed.get();
        }

        public long getInboxesProcessed() {
            return this.inboxesProcessed.get();
        }

        public long getMessagesDeleted() {
            return this.messagesDeleted.get();
        }

        public void incrementExpiredCount() {
            this.inboxesExpired.incrementAndGet();
        }

        public void incrementFailedCount() {
            this.inboxesFailed.incrementAndGet();
        }

        public void incrementProcessedCount() {
            this.inboxesProcessed.incrementAndGet();
        }

        public void incrementMessagesDeleted(long count) {
            this.messagesDeleted.addAndGet(count);
        }
    }
}

